/* Nilo - A Network boot loader.

   Copyright (C) 1999 Free Software Foundation, Inc.

This file is part of Nilo.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* This is the include generated by configuring */
#include "niloconfig.h"
#include "trace.h"

#include	<stdlib.h>
#include	<arpa/inet.h>
#include	<linux/if_ether.h>
#include	"nilo.h"
#include	"bootp.h"
#include	"tftp.h"
#include	"arp.h"
#include	"tagged.h"
#include	"printk.h"
#include	"util.h"

#undef	SIMULATE_BLOCK_LOSS

extern unsigned long		jiffies;

static struct protocol_table	*prot;
static struct bootp_t		*bootpr = 0;
static struct tftp_t		*tftpr = 0;
static long			timeout;
static int			xid;
static short			opcode = TFTP_RRQ;
static int			blockno = 0, nextblockno = 0;
static int			file_loaded = 0;
#ifdef	SIMULATE_BLOCK_LOSS
static int			losethisblock = 0;
#endif

long				seed;	/* for random number generator */

/**************************************************************************
RFC951_SLEEP - sleep for expotentially longer times
**************************************************************************/
long rfc951_sleep(int exp)
{
	long		q,z,tmo;

	/* simplified version of the LCG given in Bruce Scheier's
	   "Applied Cryptography" */
	q = seed/53668;
	if ((seed = 40014*(seed-53668*q) - 12211*q) < 0) seed += 2147483563l;
	/* compute mask */
	for (tmo = 63; tmo <= 60*1008 && --exp > 0; tmo = 2*tmo+1);
	return (tmo & seed);
}

void bootp_callback(int snaplen, struct bootp_t *bpr)
{
	printk("Bootp reply received\n");
	if (bpr->bp_xid != xid)		/* not for us */
		return;
	bootpr = bpr;
}

void process_bootp_reply(struct bootp_t *bpr)
{
	unsigned char		*p;

	printk("Client %s, ", _inet_ntoa(bpr->bp_ciaddr));
	printk("me %s, ", _inet_ntoa(bpr->bp_yiaddr));
	printk("server %s, ", _inet_ntoa(bpr->bp_siaddr));
	printk("gateway %s\n", _inet_ntoa(bpr->bp_giaddr));
	for (p = bpr->bp_vend; p < &bpr->bp_vend[200] && *p != TAG_END; ++p)
		printk("%02x ", *p);
	add_to_cache(bpr->bp_yiaddr, bpr->bp_chaddr, ARP_MYSELF);
}

static void tftp_callback(int snaplen, struct tftp_t *tpr)
{
	unsigned short		status;
#ifdef	SIMULATE_BLOCK_LOSS
	static int		donelosing = 0;
#endif

	snaplen -= sizeof(short) * 2;
	switch (status = ntohs(tpr->opcode))
	{
	case TFTP_DATA:
		printk("DATA block %d, length %d\n",
			blockno = ntohs(tpr->u.data.block),
			snaplen);
#ifdef	SIMULATE_BLOCK_LOSS
		losethisblock = 0;
		if (!donelosing && blockno == 5)
		{
			donelosing = 1;
			losethisblock = 1;
		}
#endif
		if (blockno == nextblockno + 1)	/* not a dup */
			handle_block(nextblockno = blockno,
				tpr->u.data.download, snaplen);
		else if (blockno > nextblockno)	/* skipped a block, fatal */
			;			/* what do we do? */
		opcode = TFTP_ACK;
		if (snaplen < blocksize)
			file_loaded = 1;
		break;
	case TFTP_ERROR:
		printk("ERROR %d: %s\n", ntohs(tpr->u.err.errcode),
			tpr->u.err.errmsg);
		break;
	case TFTP_OACK:
		if (strcmp(tpr->u.oack.data, "blksize") == 0)
		{
			blocksize = atol(tpr->u.oack.data + sizeof("blksize"));
			printk("OACK Setting block size to %d\n", blocksize);
		}
		opcode = TFTP_ACK;
		break;
	default:
		printk("Unknown TFTP packet type %d\n", status);
		break;
	}
}

static void jump_to_code(void)
{
}

int 
main(int argc, char **argv)
{
	int			retries;

        NILO_TRACE_FUNCTION;

	printk("NILO dated " VERSION "\n");
	init_system();
	prot = init_protocol_code();
	for (retries = 0; ; ++retries)
	{
		if (init_hardware(prot))
			break;
		timeout = jiffies + 100;	/* wait 10 seconds */
		do {
			(void)prot->wait_for_work();
		} while (jiffies < timeout);
	}
	/* Initialise seed for random number generator from MAC address */
	memcpy(&seed, &prot->macaddr[2], sizeof(seed));
#ifdef	MYTFTP
	bootpr = malloc(sizeof(struct bootp_t));
	bootpr->bp_siaddr.s_addr = htonl(0xCB0CAD88);
	strcpy(bootpr->bp_file, "/tmp/linux.nb");
#else
	for (retries = 0; ; ++retries)
	{
		timeout = jiffies + rfc951_sleep(retries);
printk("Timeout %ld\n", timeout - jiffies);
		printk("Sending bootp request\n");
		prot->bootp_handler(xid = getxid(), bootp_callback);
		do {
			(void)prot->wait_for_work();
		} while (jiffies < timeout && bootpr == 0);
		if (bootpr != 0)		/* got reply */
			break;
	}
	process_bootp_reply(bootpr);
#endif
	opcode = TFTP_RRQ;
	do {
		prot->tftp_handler(bootpr->bp_siaddr, opcode,
			bootpr->bp_file, blockno, tftp_callback);
		timeout = jiffies + 500;
		do {
#ifdef	SIMULATE_BLOCK_LOSS
			if (prot->wait_for_work() >= 0 && !losethisblock)
#else
			if (prot->wait_for_work() >= 0)
#endif
				break;
		} while (jiffies < timeout);
	} while (!file_loaded);
	jump_to_code();

        NILO_TRACE_FUNCTION_RETURN;
	return (0);
}












