/* netdrv_init.c: Initialization for network devices. */
/*
	Written 1993,1994,1995 by Donald Becker.

	The author may be reached as becker@cesdis.gsfc.nasa.gov or
	C/O Center of Excellence in Space Data and Information Sciences
		Code 930.5, Goddard Space Flight Center, Greenbelt MD 20771

	This file contains the initialization for the "pl14+" style ethernet
	drivers.  It should eventually replace most of drivers/net/Space.c.
	It's primary advantage is that it's able to allocate low-memory buffers.
	A secondary advantage is that the dangerous NE*000 netcards can reserve
	their I/O port region before the SCSI probes start.

	Modifications/additions by Bjorn Ekwall <bj0rn@blox.se>:
		ethdev_index[MAX_ETH_CARDS]
		register_netdev() / unregister_netdev()
		
	Modifications by Wolfgang Walter
		Use dev_close cleanly so we always shut things down tidily.
		
	Changed 29/10/95, Alan Cox to pass sockaddr's around for mac addresses.
	
	14/06/96 - Paul Gortmaker:	Add generic eth_change_mtu() function.

	August 12, 1996 - Lawrence V. Stefani: Added fddi_change_mtu() and
					  fddi_setup() functions.
	Sept. 10, 1996  - Lawrence V. Stefani: Increased hard_header_len to
					  include 3 pad bytes.
*/

#include "niloconfig.h"
#include "trace.h"

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/malloc.h>
#include <linux/if_ether.h>
#include <linux/string.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/fddidevice.h>
#include <linux/trdevice.h>
#include <linux/if_arp.h>
#ifdef CONFIG_NET_ALIAS
#include <linux/net_alias.h>
#endif

/* The network devices currently exist only in the socket namespace, so these
   entries are unused.  The only ones that make sense are
    open	start the ethercard
    close	stop  the ethercard
    ioctl	To get statistics, perhaps set the interface port (AUI, BNC, etc.)
   One can also imagine getting raw packets using
    read & write
   but this is probably better handled by a raw packet socket.

   Given that almost all of these functions are handled in the current
   socket-based scheme, putting ethercard devices in /dev/ seems pointless.
   
   [Removed all support for /dev network devices. When someone adds
    streams then by magic we get them, but otherwise they are un-needed
	and a space waste]
*/

/* The list of used and available "eth" slots (for "eth0", "eth1", etc.) */
#define MAX_ETH_CARDS 16 /* same as the number if irq's in irq2dev[] */
static struct device *ethdev_index[MAX_ETH_CARDS];


/* Fill in the fields of the device structure with ethernet-generic values.

   If no device structure is passed, a new one is constructed, complete with
   a SIZEOF_PRIVATE private data area.

   If an empty string area is passed as dev->name, or a new structure is made,
   a new name string is constructed.  The passed string area should be 8 bytes
   long.
 */

struct device *
init_etherdev(struct device *dev, int sizeof_priv)
{
	int new_device = 0;
	int i;

    NILO_TRACE_FUNCTION;

	/* Use an existing correctly named device in Space.c:dev_base. */
	if (dev == NULL) {
		int alloc_size = sizeof(struct device) + sizeof("eth%d  ")
			+ sizeof_priv + 3;
		struct device *cur_dev;
		char pname[8];		/* Putative name for the device.  */

		for (i = 0; i < MAX_ETH_CARDS; ++i)
			if (ethdev_index[i] == NULL) {
				sprintf(pname, "eth%d", i);
				for (cur_dev = dev_base; cur_dev; cur_dev = cur_dev->next)
					if (strcmp(pname, cur_dev->name) == 0) {
						dev = cur_dev;
						dev->init = NULL;
						sizeof_priv = (sizeof_priv + 3) & ~3;
						dev->priv = sizeof_priv
                            ? kmalloc(sizeof_priv, GFP_KERNEL)
                            :	NULL;
						if (dev->priv) memset(dev->priv, 0, sizeof_priv);
						goto found;
					}
			}
        
		alloc_size &= ~3;		/* Round to dword boundary. */
        
		dev = (struct device *)kmalloc(alloc_size, GFP_KERNEL);
		memset(dev, 0, alloc_size);
		if (sizeof_priv)
			dev->priv = (void *) (dev + 1);
		dev->name = sizeof_priv + (char *)(dev + 1);
		new_device = 1;
	}
    
 found:						/* From the double loop above. */
    
	if (dev->name &&
		((dev->name[0] == '\0') || (dev->name[0] == ' '))) {
		for (i = 0; i < MAX_ETH_CARDS; ++i)
			if (ethdev_index[i] == NULL) {
				sprintf(dev->name, "eth%d", i);
				ethdev_index[i] = dev;
				break;
			}
	}
    
	ether_setup(dev); 	/* Hmmm, should this be called here? */
	
	if (new_device) {
		/* Append the device to the device queue. */
		struct device **old_devp = &dev_base;
		while ((*old_devp)->next)
			old_devp = & (*old_devp)->next;
		(*old_devp)->next = dev;
		dev->next = 0;
	}

    NILO_TRACE_FUNCTION_RETURN;

	return dev;
}


/*
 * Local variables:
 *  compile-command: "gcc -D__KERNEL__ -I/usr/src/linux/net/inet -Wall -Wstrict-prototypes -O6 -m486 -c net_init.c"
 *  version-control: t
 *  kept-new-versions: 5
 *  tab-width: 4
 * End:
 */
