/* Nilo - A Network boot loader.

   Copyright (C) 1999 Free Software Foundation, Inc.

This file is part of Nilo.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* This is the include generated by configuring */
#include "niloconfig.h"
#include "trace.h"

#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>

#if 0           /* the old version of OSKit */
#include <flux/c/endian.h>
#else
#include "oskit/machine/endian.h"
#endif

#include "nilo.h"
#include "bootp.h"
#include "tftp.h"
#include "ip.h"
#include "arp.h"
#include "standalone.h"
#include "start.h"

extern struct sk_buff_head buff_list;

struct bootp_request
{
    struct iphdr		ip;
    struct udphdr		udp;
    struct bootp_t		bootp;
};

struct tftp_request
{
    struct iphdr		ip;
    struct udphdr		udp;
    struct tftp_t		tftp;
};

struct eth_header
{
    unsigned char		daddr[6];
    unsigned char		saddr[6];
    _u16			type;
};

/* a template iphdr so we can do a struct copy, then fill missing bits */

static struct iphdr		template_iphdr =
{
    0x45,			/* IPv4, 5 words */
    0,			/* service */
    0,			/* len, fill in later, remember htons */
    0,			/* ident */
    0,			/* frags */
    60,			/* ttl */
    IPPROTO_UDP,		/* protocol number assigned to UDP */
    0,			/* checksum, fill in later */
    0,			/* src, fill in later */
    0			/* dst, fill in later */
};

/* no template_udphdr because all fields have to be filled in anyway */

static struct protocol_table prot;
static struct bootp_t bp_rcvpkt;
static void (*bootp_callback)(int, struct bootp_t *);
static void (*tftp_callback)(int, struct tftp_t *);

static unsigned short
ipchksum(unsigned short *ip, int len)
{
    unsigned long sum = 0;
    
    len >>= 1;
    while (len--) {
        sum += *ip++;
        if (sum > 0xFFFF)
            sum -= 0xFFFF;
    }
    return((~sum) & 0xFFFF);
}

void
send_packet(struct sk_buff *sk, _u16 type)
{
    struct eth_header *eth;
    
    NILO_TRACE_FUNCTION;
    eth = (struct eth_header *)skb_push(sk, sizeof(struct eth_header));
    memcpy(eth->saddr, prot.dev->dev_addr, ETH_ALEN);
    memcpy(eth->daddr, lookup_addr(sk->raddr), ETH_ALEN);
    eth->type = htons(type);
    prot.dev->hard_start_xmit(sk, prot.dev);
    
    NILO_TRACE_FUNCTION_RETURN;
}

static void
bootp_handler(long xid, void (*callback)(int, struct bootp_t *))
{
    struct sk_buff *sk;
    struct bootp_request *bp;
    int	len;
    
    NILO_TRACE_FUNCTION;
    
    if ((sk = dev_alloc_skb(MIN_MTU)) == 0)
    {
        printk("Out of memory for BOOTP request buffer!\n");
        return;
    }
    bp = (struct bootp_request *)skb_put(sk, MIN_MTU);
    len = fill_bootp_request(&bp->bootp, prot.dev->dev_addr, xid) +
        sizeof(struct udphdr);
    bp->udp.src = 0;
    bp->udp.dest = htons(IPPORT_BOOTPS);
    bp->udp.len = htons(len);
    bp->udp.chksum = 0;	/* maybe we should do checksumming */
    bp->ip = template_iphdr;
    bp->ip.len = htons(len);
    /* bp->ip.src == 0 */
    sk->daddr = sk->raddr = bp->ip.dest = INADDR_BROADCAST;
    bp->ip.chksum = ipchksum((unsigned short *)&bp->ip, sizeof(struct iphdr));
    bootp_callback = callback;
    send_packet(sk, ETH_P_IP);
    
    NILO_TRACE_FUNCTION_RETURN;
}

static void
tftp_handler(unsigned long server, short opcode,
             const char *filename, int blockno,
             void (*callback)(int, struct tftp_t *))
{
    struct sk_buff *sk;
    struct tftp_request	*tp;
    int	len;
    
    NILO_TRACE_FUNCTION;

    if ((sk = dev_alloc_skb(MIN_MTU))== 0)
    {
        printk("Out of memory for TFTP request buffer!\n");
        return;
    }
    tp = (struct tftp_request *)skb_put(sk, MIN_MTU);
    len = fill_tftp_request(&tp->tftp, opcode, filename, 0) +
        sizeof(struct udphdr);
    tp->udp.src = 0;		/* should choose a temp one */
    tp->udp.dest = htons(IPPORT_TFTP);
    tp->udp.len = htons(len);	/* ??? */
    tp->udp.chksum = 0;	/* maybe we should do checksumming */
    tp->ip = template_iphdr;
    tp->ip.len = htons(len);
    sk->saddr = tp->ip.src = htonl(bp_rcvpkt.bp_yiaddr);
    sk->daddr = sk->raddr = tp->ip.dest = htonl(server);
    tp->ip.chksum = ipchksum((unsigned short *)&tp->ip, sizeof(struct iphdr));
    tftp_callback = callback;
    send_packet(sk, ETH_P_IP);

    NILO_TRACE_FUNCTION_RETURN;
}

static void
handle_packet(struct sk_buff *skb)
{
    NILO_TRACE_FUNCTION;
    NILO_TRACE_FUNCTION_RETURN;
}

static int
wait_for_work(void)
{
    struct sk_buff *skb;

    NILO_TRACE_FUNCTION;
        
    if ((skb = skb_dequeue(&buff_list)) != 0)
    {
        handle_packet(skb);
        kfree_skb(skb, 0);
    }
    
    NILO_TRACE_FUNCTION_RETURN;
    return (0);
}

struct protocol_table *init_protocol_code(void)
{
    NILO_TRACE_FUNCTION;
    
    /* duplication of macaddr but want to avoid referencing struct device
       in main */
    memcpy(prot.macaddr, eth0dev.dev_addr, ETH_ALEN);
    prot.dev = &eth0dev;
    prot.bpr = &bp_rcvpkt;
    prot.bootp_handler = bootp_handler;
    prot.tftp_handler = tftp_handler;
    prot.wait_for_work = wait_for_work;
    return (&prot);
    
    NILO_TRACE_FUNCTION_RETURN;
}






