/* Nilo - A Network boot loader.

   Copyright (C) 1999 Free Software Foundation, Inc.

This file is part of Nilo.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* This is the include generated by configuring */
#include "niloconfig.h"

/*
 *	This file contains the variables and routines needed by a standalone
 *	version.
 */

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/delay.h>
#include <linux/head.h>

#if 0           /* the old version of OSKit */
#include <flux/c/stdlib.h>
#include <flux/lmm.h>
#else
#include "oskit/c/stdlib.h"
#include "oskit/lmm.h"
#endif

#include "start.h"

extern unsigned long	loops_per_sec;

int		EISA_bus = 0;
/* XXXX: what is the relationship between this and bogomips? */
unsigned long	intr_count = 0;		/* inside an interrupt handler? */
unsigned long	bh_active = 0;
void		*irq2dev_map[16];
desc_table	idt;

struct nic
{
	char	*nic_name;
	int	(*nic_probe)(struct device *);
};

#ifdef	INCLUDE_3C509
extern int	el3_probe(struct device *);
#endif
#ifdef	INCLUDE_3C507
extern int	el16_probe(struct device *);
#endif
#ifdef	INCLUDE_NE
extern int	ne_probe(struct device *);
#endif
#ifdef	INCLUDE_ULTRA
extern int	ultra_probe(struct device *);
#endif
#ifdef	INCLUDE_WD
extern int	wd_probe(struct device *);
#endif

/* Edit this to add devices to probe */
struct nic	nic_list[] =
{
#ifdef	INCLUDE_3C509
	{ "3c509", el3_probe },
#endif
#ifdef	INCLUDE_3C507
	{ "3c507", el16_probe },
#endif
#ifdef	INCLUDE_NE
	{ "NE2000", ne_probe },
#endif
#ifdef	INCLUDE_ULTRA
	{ "SMC-Ultra", ultra_probe },	
#endif
#ifdef	INCLUDE_WD
	{ "SMC/WD", wd_probe },	
#endif
	{ 0, 0 }
};

struct device	eth0dev =
{
	"eth0",			/* interface name			*/
	0x0,			/* recv memory end			*/
	0x0,			/* recv memory start			*/
	0x0,			/* memory end				*/
	0x0,			/* memory start				*/
	0,			/* base I/O address			*/
	0,			/* IRQ					*/
	0, 0, 0,		/* flags				*/
	0			/* next device				*/
};

static struct device	*dev = &eth0dev;
struct device 		*dev_base = &eth0dev;

struct sk_buff_head	buff_list;

struct protocol_table;

#define		HEAPSIZE	(1024*1024)	/* 1M heap memory */

unsigned long		topmem, heapstart;
extern lmm_t		malloc_lmm;
lmm_region_t		region;

void	init_IRQ(void);
void	calibrate_delay(void);

static void timer_handler(int irq, void *dev_id, struct pt_regs *regs)
{
	++jiffies;
}

void init_system(void)
{
	/* memsize() returns k of extended memory, add bottom 1M */
	topmem = memsize() * 1024L + 0x100000;
	/* but keep below 16M because of DMA constraint */
	if (topmem > (16L * 1024L * 1024L))
		topmem = (16L * 1024L * 1024L);
	heapstart = topmem - HEAPSIZE;
#ifdef	DEBUG
	printk("Heap start = %ld, top of memory = %ld\n", heapstart, topmem);
#endif
	lmm_init(&malloc_lmm);
	lmm_add_region(&malloc_lmm, &region, (void *)0, (unsigned long)-1, 0, 0);
	lmm_add_free(&malloc_lmm, (void *)heapstart, HEAPSIZE);

	skb_queue_head_init(&buff_list);

#if	0
#ifdef	DEBUG
	printk("Initialising interrupts\n");
#endif
	init_IRQ();
#ifdef	DEBUG
	printk("Requesting timer interrupt\n");
#endif
	request_irq(0, &timer_handler, 0, "timer", 0);

#ifdef	DEBUG
	printk("Calibrating loops_per_sec\n");
#endif
	calibrate_delay();
#else	
	loops_per_sec = 10000000L;
#endif
}

void store(char *d, char *s, int n)
{
#ifdef	VERBOSE
	printk("Storing %d bytes at %08x\n", n, d);
#endif
	memcpy(d, s, n);
}

int init_hardware(struct protocol_table *prot)
{
	struct nic	*nic;
	int		status;

	printk("Probing...\n");
	for (nic = nic_list; nic->nic_name != 0; ++nic)
	{
		printk("[%s]\n", nic->nic_name);
		status = (nic->nic_probe)(dev);
		if (status == 0)
			break;
#ifdef	DEBUG
		else
			printk("%d\n", status);
#endif
	}
	udelay(1000000);
	if (nic->nic_name == 0)		/* tried all NICs */
	{
		printk("No adaptors found\n");
		return (0);
	}
	/* now open the NIC */
	(dev->open)(dev);
	udelay(1000000);
	return (1);
}

/* could we replace this with a macro expanding to constant 0? */
int check_region(unsigned int from, unsigned int extent)
{
	/* always succeed */
	return (0);
}

/* Could we replace this with macro expanding to no text? */
void request_region(unsigned int from, unsigned int extent, const char *name)
{
}

void *kmalloc(unsigned int size, int priority)
{
	return (malloc(size));
}

void kfree(void *mem)
{
	free(mem);
}

void ether_setup(struct device *dev)
{
}

void dev_tint(struct device *dev)
{
}

void netif_rx(struct sk_buff *skb)
{
	skb_queue_tail(&buff_list, skb);
}
