/* Nilo - A Network boot loader.

   Copyright (C) 1999 Free Software Foundation, Inc.

This file is part of Nilo.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* This is the include generated by configuring */
#include "niloconfig.h"
#include "trace.h"

#include "tftp.h"
#include "tagged.h"
#include "printk.h"

static union infoblock dir;
static char *segments;
static struct segheader	sh = { 0, 0, 0, 0, 0L, 0L, 0L };
static char lastseg = 0;

extern void store(char *d, char *s, int n);

static void
setup_next_segment(void)
{
    int	vlen, flags;
    char *vinfo;
    
    NILO_TRACE_FUNCTION;

    memcpy(&sh, segments, sizeof(struct segheader));
    vlen = vlength(sh.length);
    flags = sh.flags;
    lastseg = (sh.flags & 0x4) != 0;
    vinfo = segments + sizeof(struct segheader);
#ifdef	VERBOSE
    printk("Segment header:\n"
           "\tLoadaddr: %08x Imagelength: %d"
           " Memlength: %d Flags: %08x\n",
           sh.loadaddr, sh.imglength, sh.memlength, sh.flags);
    if (vlen > 0)
        printk("\tVendor info: %s\n", vinfo);
#endif
    segments += sizeof(struct segheader) + vlen;

    NILO_TRACE_FUNCTION_RETURN;
}

static void
decode_directory(char *data)
{
    int	i, vlen, flags;
    char *vinfo;
    
    NILO_TRACE_FUNCTION;

    /* make a copy of the directory block */
    memcpy(&dir, data, 512);
    if (dir.i.magic != TAG_MAGIC || dir.s[255] != 0xAA55)
    {
        printk("Not a tagged image\n");
        return;		/* should signal error and abort transfer */
    }
    vlen = vlength(dir.i.length);
    vinfo = (char *)&dir + sizeof(struct imgheader);
    segments = vinfo + vlen;
#ifdef	VERBOSE
    printk("Tagged image header:\n"
           "\tLocation: %04hx:%04hx Execaddr: %04hx:%04hx\n",
           dir.i.location.ds, dir.i.location.bx,
           dir.i.execaddr.cs, dir.i.execaddr.ip);
    if (vlen > 0)
        printk("\tVendor info: %s\n", vinfo);
#endif

    NILO_TRACE_FUNCTION_RETURN;
}

void
handle_block(int blockno, char *data, int blocksize)
{
    int	n;
    
    NILO_TRACE_FUNCTION;

    if (blockno == 1)
    {
        decode_directory(data);
        data += 512;
        blocksize -= 512;
    }
    while (blocksize > 0)
    {
        if (sh.imglength <= 0)
        {
            if (lastseg)
            {
                printk("Warning: too much data in tagged image\n");
                return;
            }
            setup_next_segment();
        }
        n = sh.imglength > blocksize ? blocksize : sh.imglength;
        store((char *)sh.loadaddr, data, n);
        sh.imglength -= n;
        sh.loadaddr += n;
        data += n;
        blocksize -= n;
    }
    /* assert blocksize == 0 here */

    NILO_TRACE_FUNCTION_RETURN;
}

