/* Nilo - A Network boot loader.

   Copyright (C) 1999 Free Software Foundation, Inc.

This file is part of Nilo.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* This is the include generated by configuring */
#include "niloconfig.h"

#include "trace.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>		/* for inet_ntoa() prototype */
#include "nilo.h"
#include "bootp.h"
#include "tftp.h"

static int bootp_socket = -1, tftp_socket = -1;
static struct sockaddr_in bootp_server, bootp_client,
				tftp_server, tftp_client;
static struct bootp_t bp_sndpkt;
static struct bootp_t bp_rcvpkt;
static struct tftp_t tftp_pkt;
unsigned char macaddr[6];
static void (*bootp_callback)(int, struct bootp_t *);
static void (*tftp_callback)(int, struct tftp_t *);

unsigned long jiffies = 0L;

/*
 *	You have to set BOOTP_SERVER and MAC_ADDR to run this version
 */

void init_system(void)
{
}

/* init_hardware --
   Initialize the hardware.
   FIXME: This should either return a void, or a real 1 or 0 
   error message.
*/ 
int
init_hardware(struct protocol_table *prot)
{
    char *p;
    int a0, a1, a2, a3, a4, a5;
    
    NILO_TRACE_FUNCTION;
    
    if ((p = getenv("MAC_ADDR")) == 0)
    {
        printf("Envvar MAC_ADDR needs to be set\n");
        exit(1);
    }

    NILO_TRACE_MSG("The Ethernet address is %s\n", p);
 
    if (sscanf(p, "%x:%x:%x:%x:%x:%x", &a0, &a1, &a2, &a3, &a4, &a5)
        < 6)
    {
        printf("MAC_ADDR is malformed\n");
        exit(1);
    }

    prot->macaddr[0] = a0;
    prot->macaddr[1] = a1;
    prot->macaddr[2] = a2;
    prot->macaddr[3] = a3;
    prot->macaddr[4] = a4;
    prot->macaddr[5] = a5;
    
    NILO_TRACE_FUNCTION_RETURN;
    return (1);
}

unsigned long getxid(void)
{
    return ((unsigned long)getpid());
}

void store(char *d, char *s, int n)
{
#ifdef	VERBOSE
    printf("Storing %d bytes at %08x\n", n, d);
#endif
}

static void
send_request(int s, void *sndbuf, int snaplen,
             struct sockaddr_in *sin_server)
{
    int i;
    
    NILO_TRACE_FUNCTION;
    
    if ((i = sendto(s, sndbuf, snaplen, 0, sin_server, sizeof(*sin_server))) < 0)
    {
        perror("sendto");
        exit(1);
    }
    if (i != snaplen)
        fprintf(stderr, "%d bytes not sent\n", snaplen);
    
    NILO_TRACE_FUNCTION_RETURN;
}

static void
bootp_init_once(void)
{
    char *server_name;
    struct hostent *server_entry;
    struct bootp_t *bootp_packet;
    static int init_done = 0;
    
    NILO_TRACE_FUNCTION;
    
    if (init_done)
        return;
    if ((server_name = getenv("BOOTP_SERVER")) == 0)
    {
        printf("Envvar BOOTP_SERVER needs to be set\n");
        exit(1);
    }
    if ((server_entry = gethostbyname(server_name)) == 0)
    {
        printf("gethostbyname cannot resolve %s\n", server_name);
        exit(1);
    }
    if ((bootp_socket = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
    {
        perror("socket");
        exit(1);
    }
    memset(&bootp_server, 0, sizeof(bootp_server));
    bootp_server.sin_family = AF_INET;
    bootp_server.sin_port = htons(IPPORT_BOOTPS);
    memcpy(&bootp_server.sin_addr, server_entry->h_addr, sizeof(bootp_server.sin_addr));
    memset(&bootp_client, 0, sizeof(bootp_client));
    bootp_client.sin_family = AF_INET;
    bootp_client.sin_port = htons(IPPORT_BOOTPC);
    bootp_client.sin_addr.s_addr = INADDR_ANY; 
    if (bind(bootp_socket, &bootp_client, sizeof(bootp_client)) < 0)
    {
        perror("bind");
        if (errno == EACCES)
            fprintf(stderr, "nilo needs to run as root\n");
        exit(1);
    }
    init_done = 1;
    
    NILO_TRACE_FUNCTION_RETURN;
}

static void
bootp_handler(long xid, void (*callback)(int, struct bootp_t *))
{
    int		snaplen;
    
    NILO_TRACE_FUNCTION;
    
    bootp_init_once();
    snaplen = fill_bootp_request(&bp_sndpkt, macaddr, xid);

    printf("Sending request from %02x:%02x:%02x:%02x:%02x:%02x to %s\n",
           macaddr[0], macaddr[1], macaddr[2],
           macaddr[3], macaddr[4], macaddr[5],
           inet_ntoa(bootp_server.sin_addr));
    send_request(bootp_socket, &bp_sndpkt, snaplen, &bootp_server);
    bootp_callback = callback;
    
    NILO_TRACE_FUNCTION_RETURN;
}

static void
tftp_init_once(unsigned long server)
{
    static int		init_done = 0;
    
    NILO_TRACE_FUNCTION;
    
    if (init_done)
        return;
    if ((tftp_socket = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
    {
        perror("socket");
        exit(1);
    }
    memset(&tftp_server, 0, sizeof(tftp_server));
    tftp_server.sin_family = AF_INET;
#ifdef	MYTFTP
    tftp_server.sin_port = htons(10069);
#else
    tftp_server.sin_port = htons(IPPORT_TFTP);
#endif
    memcpy(&tftp_server.sin_addr, &server, sizeof(tftp_server.sin_addr));
    memset(&tftp_client, 0, sizeof(tftp_client));
    tftp_client.sin_family = AF_INET;
    tftp_client.sin_port = htons(10070);
    tftp_client.sin_addr.s_addr = INADDR_ANY; 
    if (bind(tftp_socket, &tftp_client, sizeof(tftp_client)) < 0)
        perror("bind");
    init_done = 1;
}

static void
tftp_handler(unsigned long server, short opcode,
             const char *filename, int blockno,
             void (*callback)(int, struct tftp_t *))
{
    int	snaplen;
    NILO_TRACE_FUNCTION;
    
    tftp_init_once(server);
    tftp_callback = callback;
    if (opcode == TFTP_RRQ)
        printf("Requesting file %s\n", filename);
    else if (opcode == TFTP_ACK)
        printf("Ack of block %d\n", blockno);
    snaplen = fill_tftp_request(&tftp_pkt, opcode, filename, blockno);
    send_request(tftp_socket, &tftp_pkt, snaplen, &tftp_server);
    
    NILO_TRACE_FUNCTION_RETURN;
}

static int
wait_for_work(void)
{
    fd_set		rset;
    int		i, last_fd;
    struct timeval	tick;
    
//        NILO_TRACE_FUNCTION;
    
    FD_ZERO(&rset);
    if (bootp_socket > 0)
        FD_SET(bootp_socket, &rset);
    if (tftp_socket > 0)
        FD_SET(tftp_socket, &rset);
    last_fd = tftp_socket > bootp_socket ? tftp_socket : bootp_socket;
    ++last_fd;
    tick.tv_sec = 0;
    tick.tv_usec = 10000;		/* 100 Hz interrupt rate */
    if ((i = select(last_fd, &rset, 0, 0, &tick)) < 0)
    {
        perror("select");
        exit(1);
    }
    if (i == 0)
    {
        ++jiffies;
        return (-1);
    }
    if (bootp_socket > 0 && FD_ISSET(bootp_socket, &rset))
    {
        struct sockaddr_in	sin_from;
        int			len, fromsize = sizeof(sin_from);
        
        if ((len = recvfrom(bootp_socket, &bp_rcvpkt, sizeof(bp_rcvpkt), 0,
                            &sin_from, &fromsize)) < 0)
        {
            perror("recvfrom");
            exit(1);
        }
        (*bootp_callback)(len, &bp_rcvpkt);
    }
    if (tftp_socket > 0 && FD_ISSET(tftp_socket, &rset))
    {
        struct sockaddr_in	sin_from;
        int			len, fromsize = sizeof(sin_from);
        
        if ((len = recvfrom(tftp_socket, &tftp_pkt, sizeof(tftp_pkt), 0,
                            &sin_from, &fromsize)) < 0)
        {
            perror("recvfrom");
            exit(1);
        }
        tftp_server.sin_port = sin_from.sin_port;	/* change ports */
        (*tftp_callback)(len, &tftp_pkt);
    }
    
    NILO_TRACE_FUNCTION_RETURN;
    return (0);
}

struct protocol_table *init_protocol_code(void)
{
    struct protocol_table *prot;
    
    if ((prot = malloc(sizeof(*prot))) == 0)
    {
        printf("Malloc failed\n");
        exit (1);
    }
    prot->bpr = &bp_rcvpkt;
    prot->bootp_handler = bootp_handler;
    prot->tftp_handler = tftp_handler;
    prot->wait_for_work = wait_for_work;
    return (prot);
}

char *
_inet_ntoa(unsigned long a)
{
    struct in_addr i;
    
    NILO_TRACE_FUNCTION;
    
    i.s_addr = a;
    return (inet_ntoa(i));
    
    NILO_TRACE_FUNCTION_RETURN;
}
